/* $Header: /home/yav/catty/fkiss/RCS/sound.c,v 1.20 2000/09/02 03:26:02 yav Exp $
 * fkiss sound routine for Linux and FreeBSD
 * written by yav <yav@bigfoot.com>
 */

char id_sound[] = "$Id: sound.c,v 1.20 2000/09/02 03:26:02 yav Exp $";

#include <stdio.h>
#include "config.h"
#include "headers.h"

#ifdef FKISS_CONFIG_H
# include <X11/Xos.h>
#else
# ifdef HAVE_FCNTL_H
#  include <fcntl.h>
# endif
#endif

#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#else
# ifdef HAVE_STAT_H
#  include <stat.h>
# endif
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef linux
# define USE_DSP
# define DEV_SOUND	"/dev/dsp"
# ifdef HAVE_FCNTL_H
#   include <fcntl.h>
# endif
# ifdef HAVE_SYS_IOCTL_H
#   include <sys/ioctl.h>
# endif
# include <linux/soundcard.h>
#endif

#ifdef __FreeBSD__
# define USE_DSP
# define DEV_SOUND	"/dev/dsp"
# include <machine/soundcard.h>
#endif

#ifndef DEV_SOUND
# define DEV_SOUND	"/dev/audio"
#endif

char *sound_device = DEV_SOUND;
int sound_debug = 1;		/* debug information print level
				 * 0: print nothing, silent
				 * 1: error messages
				 * 2: all messages
				 */
int sound_output = 1;		/* 0:Not output to device, other:output */
static int sound_enable = 0;	/* 0:disable other:enable */

int sound_rate = 22050;		/* sampling rate Hz */
int sound_channels = 1;		/* 1:mono 2:stereo */
int sound_bits = 8;		/* bits per sample 8 or 16 */
int sound_ulaw_encoded = 0;	/* input ulaw encoded */
int sound_little_endian = 0;	/* input little-endian */
int sound_unsigned = -1;	/* input zero level 0x80 */
int sound_force = 0;		/* no conversion, force write DSP data */

#define SOUND_CACHE_MAX 64
static int cache_n;
typedef struct {
  char *pathname;
  char *buffer;
  long size;
} CACHE_TABLE;
static CACHE_TABLE *cache_list = NULL;

#ifdef USE_DSP

static int sample_bytes;	/* byts per sample 1 or 2 */
static int dsp_rate;
static int dsp_bits;
static int dsp_channels;

#include "ulaw.h"

#endif

void adjust_parms()
{
  if (sound_bits < 1)
    sound_bits = 8;
#ifdef USE_DSP
  sample_bytes = (sound_bits+7) / 8;
  /* defaut 8bit data zero level 0x80 */
  if (sound_unsigned < 0)
    sound_unsigned = (sample_bytes == 1);
#endif
}

#ifdef USE_DSP

int setup_dsp(fd)
     int fd;
{
  /* copy to DSP setting parms */
  dsp_channels = sound_channels;
  dsp_bits = sound_bits;
  dsp_rate = sound_rate;
  ioctl(fd, SOUND_PCM_WRITE_CHANNELS, &dsp_channels);
  ioctl(fd, SOUND_PCM_WRITE_BITS, &dsp_bits);
  ioctl(fd, SOUND_PCM_WRITE_RATE, &dsp_rate);
  return 0;
}

void conv_center80(buf, n)
     unsigned char *buf;
     int n;
{
  while (n--) {
    *buf += 0x80;
    buf++;
  }
}

/* convert ulaw to  0x80 center 8bit linear PCM */
void conv_ulaw_dsp(buf, n)
     unsigned char *buf;
     int n;
{
  while (n--)
    *buf++ = ulaw_dsp[*buf];
}

int get_highest_bytes(buf, n)
     unsigned char *buf;
     int n;
{
  int i, d;
  unsigned char *p;
  unsigned char *p0;
  
  d = sample_bytes * sound_channels;
  p0 = p = buf;
  if (sound_little_endian)
    p0 += sample_bytes-1;
  if (sound_channels == 1) {
    for (i = 0; i < n; i += d) {
      *p++ = *p0;
      p0 += d;
    }
  } else {
    for (i = 0; i < n; i += d) {
      *p++ = *p0;
      *p++ = *(p0+sample_bytes);
      p0 += d;
    }
  }
  return (n / sample_bytes);
}

#endif /* USE_DSP */

int convert_sound(buf, n)
     char *buf;
     int n;
{
#ifdef USE_DSP
  if (sound_force)
    return n;
  if (sound_ulaw_encoded) {
    conv_ulaw_dsp(buf, n);
    return n;
  }
  if (dsp_bits == 8) {
    if (sample_bytes != 1)
      n = get_highest_bytes(buf, n);
    /* adjust zero level 0x80 */
    if (!sound_unsigned)
      conv_center80(buf, n);
  }
#endif
  return n;
}

#ifdef USE_DSP
void play_sync(fd)
     int fd;
{
  if (sound_debug >= 2)
    fputs("sync", stderr);
  ioctl(fd, SNDCTL_DSP_SYNC);
  if (sound_debug >= 2)
    fputs(" ", stderr);
}
#endif

int play_sound(fd, buf, n)
     int fd;
     char *buf;
     int n;
{
#ifdef USE_DSP
  play_sync(fd);
#endif
  if (sound_debug >= 2)
    fputs("write", stderr);
  if (sound_output)
    write(fd, buf, n);
  if (sound_debug >= 2)
    fputs(" ", stderr);
  return 0;
}

short get_short(p)
     unsigned char *p;
{
  return sound_little_endian ?
    (*(p+1) << 8) | *p :
  (*p << 8) | *(p+1);
}

long get_long(p)
     unsigned char *p;
{
  return sound_little_endian ?
    (((long)*(p+3)) << 24) | (((long)*(p+2)) << 16) | (*(p+1) << 8) | *p :
  (((long)*p) << 24) | (((long)*(p+1)) << 16) | (*(p+2) << 8) | *(p+3);
}

/* check Sun Audio file header
 * return data offset to play
 *        == 0  : Not au file
 *        <  0  : Not supported format
 */
int check_au_header(buf)
     unsigned char *buf;	/* data buffer */
{
  long data_offset;
  long data_type;
  static struct {
    int bits;			/* bits per sample */
    int ulaw;			/* 0:linear 1:ulaw encoded */
  } type_info[] = {
    { 0, 0},			/* 0: dummy */
    { 8, 1},			/* 1: 8-bit ISDN u-law */
    { 8, 0},			/* 2: 8-bit linear PCM */
    {16, 0},			/* 3:16-bit linear PCM */
    {24, 0},			/* 4:24-bit linear PCM */
    {32, 0},			/* 5:32-bit linear PCM */
  };
  
  sound_little_endian = 0;
  if (strncmp(buf, ".snd", 4) != 0) {
    if (strncmp(buf, "dns.", 4) != 0)
      return 0;			/* Not au file! */
    sound_little_endian = 1;
  }
  data_type    = get_long(buf+0x0c);
  if (data_type < 1 || data_type > 5)
    return -1;			/* Not supported format */
  data_offset  = get_long(buf+0x04);
  /*  data_size    = get_long(buf+0x08); */
  sound_rate     = get_long(buf+0x10);
  sound_channels = get_long(buf+0x14);

  sound_bits = type_info[data_type].bits;
  sound_ulaw_encoded = type_info[data_type].ulaw;
  return data_offset;
}


/* check Microsoft RIFF-WAVE header
 * return data offset
 *        == 0 : Not wav file
 *        < 0  : Not supported format
 */
int check_wav_header(buf, n)
     unsigned char *buf;
     int n;
{
  int i;
  long data_rate;
  long data_channel;
  unsigned char *p;
  long len;
  
  sound_little_endian = 1;
  if (strncmp(buf, "RIFF", 4))
    return 0;			/* Not wav file */
  len = get_long(buf+4);
  if (strncmp(buf+8, "WAVE", 4))
    return 0;			/* Not wav file */

  p = buf+12;
  if ((n -= 12) < 0)
    return 0;
  while (strncmp(p, "fmt ", 4) != 0) {
    len = get_long(p+4);
    p += 8 + len;
    if ((n -= 8 + len) < 0)
      return 0;
  }
  len = get_long(p+4);
  
  p += 8;
  if ((n -= 8) < 0)
    return 0;
  i = get_short(p);
  switch (i) {
  case 0x0001:			/* WAVE_FORMAT_PCM */
    break;
  case 0x0000:			/* WAVE_FORMAT_UNKNOWN */
  case 0x0002:			/* WAVE_FORMAT_ADPCM */
  case 0x0006:			/* WAVE_FORMAT_ALAW */
  case 0x0007:			/* WAVE_FORMAT_MULAW */
  case 0x0010:			/* WAVE_FORMAT_OKI_ADPCM */
  case 0x0015:			/* WAVE_FORMAT_DIGISTD */
  case 0x0016:			/* WAVE_FORMAT_DIGIFIX */
  case 0x0101:			/* IBM_FORMAT_MULAW */
  case 0x0102:			/* IBM_FORMAT_ALAW */
  case 0x0103:			/* IBM_FORMAT_ADPCM */
  default:
    if (sound_debug)
      fprintf(stderr, "RIFF-WAVE format 0x%04x not supported!\n", i);
    return -1;			/* Not supported format */
  }
  data_channel = get_short(p+2);
  data_rate = get_long(p+4);
  /* long: average bytes/second */
  /* short: block align */
  i = get_short(p+14);		/* bits/sample? */
  if (sound_debug >= 2)
    fprintf(stderr, "RIFF-WAVE %ld-channels, %ldHz, %d-bits/sample\n",
	    data_channel, data_rate, i);
  switch (i) {
  case 8:
    i = 1; sound_bits = 8; sound_unsigned = 1;
    break;
  case 16:
    i = 2; sound_bits = 16; sound_unsigned = 0;
    break;
  case 32:
    i = 4; sound_bits = 32; sound_unsigned = 0;
    break;
  default:
    if (sound_debug)
      fprintf(stderr, "RIFF-WAVE 0x%04x-bits/sample not supported!\n", i);
    return -1;			/* Not supported format */
  }
  p += len;
  if ((n -= len) < 0)
    return 0;
  
  while (strncmp(p, "data", 4)) {
    len = get_long(p+4);
    if (sound_debug >= 2)
      fprintf(stderr, "%08x %ld bytes section [%.4s] skip\n", p-buf, len, p);
    p += 8 + len;
    if ((n -= 8 + len) < 0)
      return 0;
  }
  
  /* number of samples = get_long(p+4) / i */
  sound_channels = data_channel;
  sound_rate = data_rate;
  sound_ulaw_encoded = 0;
  p += 8;
  if ((n -= 8) < 0)
    return 0;
  if (sound_debug >= 2)
    fprintf(stderr, "RIFF-WAVE offset %d\n", p - buf);
  return p - buf;		/* return data offset */
}

void disp_file_info(name)
     char *name;
{
  fprintf(stderr, "device: %s\n", sound_device);
  fprintf(stderr, "input: %s\n", name == NULL ? "(stdin)" : name);
}

void disp_sound_info(name)
     char *name;
{
  fprintf(stderr, " %5dHz %2d-bit %d-channel",
	  sound_rate, sound_bits, sound_channels);
  if (sound_ulaw_encoded)
    fprintf(stderr, " u-law");
  if (sound_little_endian)
    fprintf(stderr, " Little-endian");
  if (!sound_ulaw_encoded)
    fprintf(stderr, " %s", sound_unsigned ? "unsigned" : "signed");
  fprintf(stderr, "\n");
}

#ifdef USE_DSP
void disp_dsp_info()
{
  fprintf(stderr, " %5dHz %2d-bit %d-channel\n",
	  dsp_rate, dsp_bits, dsp_channels);
}

#endif /* USE_DSP */

/* Check sound device
 * return 0:OK other:NG
 */
int sound_init()
{
  int fd;
  struct stat st;
  
  cache_n = 0;
  /* sound device exist? */
  if (stat(sound_device, &st) != 0)
    return 1;			/* device not found */
  
  /* sound device writeble? */
  fd = open(sound_device, O_WRONLY|O_NONBLOCK);
  if (fd < 0)
    return 2;			/* device open error */
  
  fcntl(fd, F_SETFL, ~O_NONBLOCK);
  close(fd);
  sound_enable = 1;
  cache_list = (CACHE_TABLE *)malloc(sizeof(CACHE_TABLE)*SOUND_CACHE_MAX);
  return 0;
}

void sound_end()
{
  int i;
  
  for (i = 0; i < cache_n; i++)
    free((cache_list+i)->buffer);
  if (cache_list != NULL)
    free(cache_list);
}

/* return audio file size
 *  minus : file not found
 */
long sound_filesize(name)
     char *name;
{
  long r;
  struct stat st;
  
  r = -1;
  if (sound_enable && stat(name, &st) == 0)
    r = st.st_size;
  return r;
}

int cached_no(name)
     char *name;
{
  int i;
  
  for (i = 0; i < cache_n; i++)
    if (strcmp(name, (cache_list+i)->pathname) == 0)
      return i;
  return -1;			/* not cached */
}

/* cache audio data
 * return minus cannot to cache
 */
int sound_cache(name)
     char *name;
{
  char *p;
  FILE *fp;
  CACHE_TABLE *cp;
  
  if (!sound_enable)
    return -1;
  if (cached_no(name) >= 0)
    return 0;			/* already cached */
  if (cache_n >= SOUND_CACHE_MAX)
    return -1;
  cp = cache_list + cache_n;
  cp->size = sound_filesize(name);
  if (cp->size < 0)
    return -1;
  fp = fopen(name, "rb");
  if (fp == NULL)
    return -1;			/* Why? stat success, but fopen fail??? */
  /* If name will be free, use strdup to keep string */
  cp->pathname = name;
  p = malloc(cp->size);
  if (p == NULL) {
    fclose(fp);
    return -1;
  }
  fread(p, cp->size, 1, fp);
  cp->buffer = p;
  cache_n++;
  fclose(fp);
  if (sound_debug >= 2)
    fprintf(stderr, "sound_cache: ``%s'' cached.\n", name);
  return 0;
}

/* Play audio file
 * return 0: No error 1:file open error 2:device open error
 */
int sound_play(name)
     char *name;		/* audio filename */
{
  int i, fd;
  int cn;
  int loadbytes;
  long left;
  int ofs;
  FILE *fp;
  char *p;
  char buf[32*1024];		/* 8 * N bytes */
  
  if (sound_enable) {
    bzero(buf, sizeof(buf));
    if (name == NULL) {
      fp = stdin;
      loadbytes = fread(buf, 1, sizeof(buf), fp);
    } else {
      cn = cached_no(name);
      if (cn >= 0) {
	fp = NULL;
	left = (cache_list+cn)->size;
	loadbytes = left > sizeof(buf) ? sizeof(buf) : left;
	p = (cache_list+cn)->buffer;
	bcopy(p, buf, loadbytes);
      } else {
	fp = fopen(name, "rb");
	if (fp == NULL)
	  return 1;		/* Sound file open error */
	loadbytes = fread(buf, 1, sizeof(buf), fp);
      }
    }
    if (sound_debug >= 2)
      disp_file_info(name);
    ofs = check_wav_header(buf, loadbytes);
    if (ofs == 0)
      ofs = check_au_header(buf);
    if (ofs < 0) {
      if (fp != NULL)
	fclose(fp);
      return 3;			/* not supported format */
    }
    adjust_parms();
    if (sound_debug >= 2)
      disp_sound_info(name);
    fd = open(sound_device, O_WRONLY|O_NONBLOCK);
    if (fd < 0) {
      if (fp != NULL)
	fclose(fp);
      return 2;			/* device open error */
    }
    fcntl(fd, F_SETFL, ~O_NONBLOCK);
#ifdef USE_DSP
    setup_dsp(fd);
    if (sound_debug >= 2)
      disp_dsp_info();
#endif
    while (loadbytes) {
      i = convert_sound(buf+ofs, loadbytes-ofs);
      if (sound_debug >= 2)
	fprintf(stderr, "%d ", i);
      play_sound(fd, buf+ofs, i);
      ofs = 0;
      if (fp == NULL) {
	left -= loadbytes;
	p += loadbytes;
	loadbytes = left > sizeof(buf) ? sizeof(buf) : left;
	bcopy(p, buf, loadbytes);
      } else {
	loadbytes = fread(buf, 1, sizeof(buf), fp);
      }
    }
    if (sound_debug >= 2)
      fputs("fin ", stderr);
#ifdef USE_DSP
    play_sync(fd);
#endif
    if (sound_debug >= 2)
      fputs("close", stderr);
    close(fd);
    if (sound_debug >= 2)
      fputs(" \n", stderr);
    if (fp != NULL)
      fclose(fp);
  }
  return 0;
}

/* End of file */
